@extends('layouts.frontend')

@section('title', 'Book a Consultation - ' . $site->company_name)

@section('content')
    <div class="bg-primary-50 min-h-screen py-16">
        <div class="container mx-auto px-4">
            <div class="max-w-2xl mx-auto bg-white rounded-2xl shadow-xl overflow-hidden">

                <div class="bg-primary-600 p-8 text-center text-white">
                    <h1 class="text-3xl font-bold">Schedule Your Session</h1>
                    <p class="mt-2 opacity-90">Choose your service and preferred time.</p>
                </div>

                <div class="p-8">
                    <form action="{{ route('booking.store') }}" method="POST" enctype="multipart/form-data" class="space-y-6">
                        @csrf

                        {{-- // if there are no active services and consultant, show a message --}}
                        @if ($services->isEmpty() || $consultants->isEmpty())
                            <div class="bg-red-50 p-4 rounded-lg text-red-700">
                                <p>Bookings are temporarily unavailable. Please check back later or call
                                    {{ $site->contact_phone }}.</p>
                            </div>
                        @else
                            <div>
                                <label class="block text-gray-700 font-bold mb-2">Select Service</label>
                                <select name="service_id"
                                    class="w-full border-gray-300 rounded-lg shadow-sm focus:border-primary-500 focus:ring focus:ring-primary-200 p-3 bg-gray-50">
                                    @foreach ($services as $service)
                                        <option value="{{ $service->id }}"
                                            {{ isset($selectedService) && $selectedService->id == $service->id ? 'selected' : '' }}>
                                            {{ $service->name }} (NPR {{ number_format($service->price, 0) }})
                                        </option>
                                    @endforeach
                                </select>
                            </div>

                            <div>
                                <label class="block text-gray-700 font-bold mb-2">Choose Consultant</label>
                                <select name="consultant_id" id="consultant_id"
                                    class="w-full border-gray-300 rounded-lg shadow-sm focus:border-primary-500 focus:ring focus:ring-primary-200 p-3 bg-gray-50">
                                    @foreach ($consultants as $consultant)
                                        <option value="{{ $consultant->id }}" {{ old('consultant_id') == $consultant->id ? 'selected' : '' }}>
                                            {{ $consultant->user->name }} - {{ $consultant->specialization }}
                                        </option>
                                    @endforeach
                                </select>
                            </div>

                            <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                                <div>
                                    <label class="block text-gray-700 font-bold mb-2">Preferred Date</label>
                                    <input type="date" name="booking_date" id="booking_date" required min="{{ date('Y-m-d') }}"
                                        value="{{ old('booking_date', date('Y-m-d')) }}"
                                        class="w-full border-gray-300 rounded-lg shadow-sm focus:border-primary-500 focus:ring focus:ring-primary-200 p-3">
                                    @error('booking_date') <p class="text-red-500 text-sm mt-1">{{ $message }}</p> @enderror
                                </div>
                                <div>
                                    <label class="block text-gray-700 font-bold mb-2">Preferred Time</label>
                                    <select name="booking_time" id="booking_time" required
                                        class="w-full border-gray-300 rounded-lg shadow-sm focus:border-primary-500 focus:ring focus:ring-primary-200 p-3 bg-white">
                                        <option value="">Loading available times...</option>
                                    </select>
                                    <p id="slot_help" class="text-xs text-gray-500 mt-1"></p>
                                    @error('booking_time') <p class="text-red-500 text-sm mt-1">{{ $message }}</p> @enderror
                                </div>
                            </div>

                            <div class="bg-gray-50 p-6 rounded-xl border border-gray-200">
                                <h3 class="font-bold text-gray-800 mb-4">Payment Required</h3>

                                <div class="flex flex-col md:flex-row gap-6 items-center mb-6">
                                    <div class="bg-white p-2 rounded shadow">
                                        @if (!empty($site->payment_qr_image))
                                            <a href="{{ asset('storage/' . $site->payment_qr_image) }}" target="_blank">
                                                <img src="{{ asset('storage/' . $site->payment_qr_image) }}"
                                                    class="w-48 h-48 object-contain" alt="Payment QR">
                                            </a>
                                            <p class="mt-2 text-xs text-gray-400 text-center">
                                                Click the QR to open full size or download.
                                            </p>
                                        @else
                                            <img src="{{ asset('storage/placeholder-image.jpg') }}"
                                                class="w-48 h-48 object-contain" alt="Payment QR">
                                        @endif
                                    </div>
                                    <div class="text-sm text-gray-600 space-y-1">
                                        @if (!empty($site->payment_bank_name))
                                            <p><strong>Bank:</strong> {{ $site->payment_bank_name }}</p>
                                        @endif
                                        @if (!empty($site->payment_account_number))
                                            <p><strong>Acct:</strong> {{ $site->payment_account_number }}</p>
                                        @endif
                                        @if (!empty($site->payment_account_name))
                                            <p><strong>Name:</strong> {{ $site->payment_account_name }}</p>
                                        @endif
                                        @if (!empty($site->payment_instructions))
                                            <p class="mt-2 text-primary-600 font-bold">
                                                {!! nl2br(e($site->payment_instructions)) !!}
                                            </p>
                                        @else
                                            <p class="mt-2 text-primary-600 font-bold">
                                                Please upload the payment screenshot below.
                                            </p>
                                        @endif
                                    </div>
                                </div>

                                <label class="block text-gray-700 font-bold mb-2">Upload Payment Screenshot</label>
                                <input type="file" name="payment_proof" required accept="image/*"
                                    class="w-full border border-gray-300 rounded-lg p-2 bg-white">
                            </div>

                            <x-primary-button class="w-full justify-center py-4 text-lg">
                                Confirm Booking Request
                            </x-primary-button>
                        @endif
                    </form>
                </div>
            </div>

            <p class="text-center text-gray-500 mt-8 text-sm">
                Need help? Call us at <span class="font-bold">{{ $site->contact_phone }}</span>
            </p>
        </div>
    </div>
@endsection

@push('scripts')
    <script>
        document.addEventListener('DOMContentLoaded', () => {
            const consultantEl = document.getElementById('consultant_id');
            const dateEl = document.getElementById('booking_date');
            const timeEl = document.getElementById('booking_time');
            const helpEl = document.getElementById('slot_help');

            if (!consultantEl || !dateEl || !timeEl) return;

            const slotsUrl = "{{ route('booking.slots') }}";
            const oldTime = @json(old('booking_time'));

            const setLoading = (text) => {
                timeEl.innerHTML = `<option value="">${text}</option>`;
                helpEl.textContent = '';
            };

            const loadSlots = async () => {
                const consultantId = consultantEl.value;
                const date = dateEl.value;

                if (!consultantId || !date) {
                    setLoading('Select date to see times');
                    return;
                }

                setLoading('Loading available times...');

                const url = new URL(slotsUrl, window.location.origin);
                url.searchParams.set('consultant_id', consultantId);
                url.searchParams.set('date', date);

                try {
                    const res = await fetch(url.toString(), { headers: { 'Accept': 'application/json' } });
                    if (!res.ok) throw new Error('Failed to load slots');
                    const data = await res.json();

                    const slots = Array.isArray(data.slots) ? data.slots : [];
                    if (slots.length === 0) {
                        timeEl.innerHTML = `<option value="">No slots available</option>`;
                        helpEl.textContent = 'Try another date or consultant.';
                        return;
                    }

                    const options = ['<option value="">Select a time</option>']
                        .concat(slots.map(t => `<option value="${t}">${t}</option>`))
                        .join('');
                    timeEl.innerHTML = options;

                    if (oldTime) {
                        timeEl.value = oldTime;
                    }
                    helpEl.textContent = `${slots.length} slots available.`;
                } catch (e) {
                    timeEl.innerHTML = `<option value="">Unable to load slots</option>`;
                    helpEl.textContent = 'Please refresh the page.';
                }
            };

            consultantEl.addEventListener('change', loadSlots);
            dateEl.addEventListener('change', loadSlots);

            loadSlots();
        });
    </script>
@endpush
