<?php

namespace Database\Seeders;

use App\Models\User;
use App\Models\Service;
use App\Models\Consultant;
use App\Models\CompanySetting;
use App\Models\NavigationItem;
use App\Models\Horoscope;
use Illuminate\Database\Seeder;
use Illuminate\Support\Facades\Hash;
use Spatie\Permission\Models\Role; // Add this

class VedicPlatformSeeder extends Seeder
{
    public function run(): void
    {
        // NOTE: This seeder is designed to be re-runnable (idempotent).

        // 1) Roles (Spatie Permission)
        Role::findOrCreate('super_admin', 'web');
        Role::findOrCreate('panel_user', 'web'); // consultants / staff
        Role::findOrCreate('client', 'web');     // frontend users

        // 2) Company Settings (single row used by $site in views)
        CompanySetting::firstOrCreate(
            ['id' => 1],
            [
                'company_name' => 'Vedic Platform',
                'contact_email' => 'admin@vedic.com',
                'contact_phone' => '+977-9800000000',
                'address' => 'Kathmandu, Nepal',
                'meta_description' => 'Vedic astrology consultations, vastu guidance, and sacred services.',
                'privacy_policy' => 'Add your privacy policy here.',
                'terms_of_service' => 'Add your terms and conditions here.',
                'payment_bank_name' => 'Nabil Bank',
                'payment_account_number' => '1234-5678-90XX',
                'payment_account_name' => 'Vedic Platform Pvt Ltd',
                'payment_instructions' => "Please scan the QR and upload your payment screenshot.\nWe confirm bookings within 24 hours.",
            ]
        );

        // 3) Navigation (dynamic menu)
        $nav = [
            ['label' => 'Home', 'url' => '/', 'order' => 1],
            ['label' => 'Services', 'url' => '/services', 'order' => 2],
            ['label' => 'Daily Horoscope', 'url' => '/horoscopes', 'order' => 3],
            ['label' => 'Blog', 'url' => '/blog', 'order' => 4],
            ['label' => 'About', 'url' => '/about', 'order' => 5],
            ['label' => 'Contact', 'url' => '/contact', 'order' => 6],
        ];

        foreach ($nav as $item) {
            NavigationItem::updateOrCreate(
                ['label' => $item['label'], 'parent_id' => null],
                [
                    'url' => $item['url'],
                    'order' => $item['order'],
                    'is_active' => true,
                ]
            );
        }

        // 4) Super Admin user (Filament)
        $adminEmail = env('VEDIC_SEED_ADMIN_EMAIL', 'admin@vedic.com');
        $adminPassword = env('VEDIC_SEED_ADMIN_PASSWORD', 'password');

        $admin = User::updateOrCreate(
            ['email' => $adminEmail],
            [
                'name' => 'Super Admin',
                'password' => Hash::make($adminPassword),
                'email_verified_at' => now(),
            ]
        );
        if (! $admin->hasRole('super_admin')) {
            $admin->assignRole('super_admin');
        }

        // 5) Consultants (panel users)
        $consultants = [
            [
                'name' => 'Acharya Shastri',
                'email' => 'shastri@vedic.com',
                'specialization' => 'Vedic Astrology',
                'experience_years' => 15,
                'rating' => 4.9,
                'languages' => ['Nepali', 'Hindi', 'Sanskrit'],
                'bio' => 'Expert in Kundali reading and traditional remedies.',
            ],
            [
                'name' => 'Pandit Sharma',
                'email' => 'sharma@vedic.com',
                'specialization' => 'Vastu & Karmakanda',
                'experience_years' => 10,
                'rating' => 4.8,
                'languages' => ['Nepali', 'Hindi'],
                'bio' => 'Specialist in Vastu guidance and sacred rituals.',
            ],
        ];

        foreach ($consultants as $c) {
            $user = User::updateOrCreate(
                ['email' => $c['email']],
                [
                    'name' => $c['name'],
                    'password' => Hash::make(env('VEDIC_SEED_CONSULTANT_PASSWORD', 'password')),
                    'email_verified_at' => now(),
                ]
            );

            if (! $user->hasRole('panel_user')) {
                $user->assignRole('panel_user');
            }

            Consultant::updateOrCreate(
                ['user_id' => $user->id],
                [
                    'bio' => $c['bio'],
                    'specialization' => $c['specialization'],
                    'experience_years' => $c['experience_years'],
                    'languages' => $c['languages'],
                    'rating' => $c['rating'],
                    'is_available' => true,
                ]
            );
        }

        // 6) Services (must match enum: astrology/vastu/karmanda)
        $services = [
            [
                'name' => 'Detailed Birth Chart Analysis',
                'slug' => 'birth-chart-analysis',
                'category' => 'astrology',
                'description' => 'A comprehensive review of your planetary positions (career, health, finance, marriage).',
                'price' => 1500.00,
                'duration_minutes' => 45,
                'is_active' => true,
            ],
            [
                'name' => 'Home Vastu Audit',
                'slug' => 'home-vastu-audit',
                'category' => 'vastu',
                'description' => 'Balance the energy of your living space with practical Vastu guidance.',
                'price' => 5000.00,
                'duration_minutes' => 60,
                'is_active' => true,
            ],
            [
                'name' => 'Navagraha Shanti Puja',
                'slug' => 'navagraha-shanti-puja',
                'category' => 'karmanda',
                'description' => 'Traditional puja performed to harmonize planetary influences.',
                'price' => 8000.00,
                'duration_minutes' => 90,
                'is_active' => true,
            ],
        ];

        foreach ($services as $svc) {
            Service::updateOrCreate(['slug' => $svc['slug']], $svc);
        }

        // 7) Minimal horoscope seeds (optional but helpful for demo)
        // Creates a daily horoscope for Aries if none exists for today.
        $today = now()->toDateString();
        Horoscope::firstOrCreate(
            [
                'zodiac_sign' => 'aries',
                'type' => 'daily',
                'valid_from' => $today,
                'valid_to' => $today,
            ],
            [
                'prediction' => 'Today brings clarity and steady progress. Focus on one priority and follow through.',
                'lucky_number' => 7,
                'lucky_color' => 'Red',
                'love' => 'Speak honestly and gently; small gestures matter.',
                'career' => 'A pending task clears faster than expected—finish it.',
                'finance' => 'Avoid impulsive purchases; plan for a bigger goal.',
                'health' => 'Stay hydrated and take short breaks from screens.',
            ]
        );
    }
}
