<?php

namespace App\Services;

use Illuminate\Support\Facades\Http;
use Carbon\Carbon;
use Illuminate\Support\Facades\Log;

class AstrologyService
{
    protected $apiKey;
    protected $baseUrl;

    public function __construct()
    {
        $this->apiKey = env('ASTROLOGY_API_KEY');
        $this->baseUrl = env('ASTROLOGY_API_URL');
    }

    public function getChartData($dob, $tob, $pob)
    {
        $coords = $this->getCoordinates($pob);
        $date = Carbon::parse($dob . ' ' . $tob);

        $payload = [
            "year" => $date->year,
            "month" => $date->month,
            "date" => $date->day,
            "hours" => $date->hour,
            "minutes" => $date->minute,
            "seconds" => 0,
            "latitude" => $coords['lat'],
            "longitude" => $coords['lon'],
            "timezone" => 5.75,
            "config" => [
                "observation_point" => "topocentric",
                "ayanamsha" => "lahiri"
            ]
        ];

        try {
            $response = Http::withHeaders([
                'Content-Type' => 'application/json',
                'x-api-key' => $this->apiKey
            ])->post($this->baseUrl . '/planets', $payload);

            if ($response->successful()) {
                $data = $response->json();

                // The API returns an array. Index 1 contains the easiest format {"Sun": {}, "Moon": {}}
                // We default to Index 1, but fallback to 0 if structure changes
                return $data['output'][1] ?? null;
            } else {
                Log::error('Astrology API Error: ' . $response->body());
                return null;
            }

        } catch (\Exception $e) {
            Log::error('Astrology Connection Error: ' . $e->getMessage());
            return null;
        }
    }

    private function getCoordinates($city)
    {
        // Default Kathmandu
        $default = ['lat' => 27.7172, 'lon' => 85.3240];

        if (empty($city)) return $default;

        try {
            $response = Http::withHeaders(['User-Agent' => 'LaravelApp/1.0'])
                ->get('https://nominatim.openstreetmap.org/search', [
                    'q' => $city,
                    'format' => 'json',
                    'limit' => 1
                ]);

            if ($response->successful() && !empty($response->json())) {
                $data = $response->json()[0];
                return ['lat' => (float) $data['lat'], 'lon' => (float) $data['lon']];
            }
        } catch (\Exception $e) {
            return $default;
        }
        return $default;
    }
}
