<?php

namespace App\Observers;

use App\Models\Booking;
use App\Models\CompanySetting;
use Illuminate\Support\Facades\Mail;
use App\Mail\BookingStatusChangedMail;
use App\Mail\NewBookingAdminMail;
use App\Mail\BookingRequestedMail;
use App\Mail\BookingRescheduledMail;
use App\Mail\AdminBookingUpdatedMail;

class BookingObserver
{
    protected function adminEmail(): string
    {
        $settings = CompanySetting::first();
        return $settings?->contact_email ?: 'admin@vedickarmanda.com';
    }

    /**
     * Handle the Booking "created" event.
     */
    public function created(Booking $booking): void
    {
        Mail::to($this->adminEmail())->send(new NewBookingAdminMail($booking));
        Mail::to($booking->user->email)->send(new BookingRequestedMail($booking));
    }

    /**
     * Handle the Booking "updated" event.
     */
    public function updated(Booking $booking): void
    {
        // Status changes (confirmed/cancelled) - notify user + admin.
        if ($booking->wasChanged('status') && in_array($booking->status, ['confirmed', 'cancelled'])) {
            Mail::to($booking->user->email)->send(new BookingStatusChangedMail($booking));
            Mail::to($this->adminEmail())->send(new AdminBookingUpdatedMail($booking, 'status'));
        }

        // Meeting link added/changed - notify user (and admin for record).
        if ($booking->wasChanged('meeting_link') && $booking->status === 'confirmed' && !empty($booking->meeting_link)) {
            Mail::to($booking->user->email)->send(new BookingStatusChangedMail($booking));
            Mail::to($this->adminEmail())->send(new AdminBookingUpdatedMail($booking, 'meeting_link'));
        }

        // Reschedule (scheduled_at changed) - notify user + admin.
        if ($booking->wasChanged('scheduled_at')) {
            Mail::to($booking->user->email)->send(new BookingRescheduledMail($booking));
            Mail::to($this->adminEmail())->send(new AdminBookingUpdatedMail($booking, 'scheduled_at'));
        }
    }
}
