<?php

namespace App\Filament\Resources;

use App\Filament\Resources\ConsultantResource\Pages;
use App\Models\Consultant;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Filament\Forms\Components\FileUpload; 
use Filament\Tables\Columns\ImageColumn; 

class ConsultantResource extends Resource
{
    protected static ?string $model = Consultant::class;
    protected static ?string $navigationIcon = 'heroicon-o-academic-cap';

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Forms\Components\Select::make('user_id')
                    ->label('Select user to make Consultant')
                    ->relationship('user', 'name')
                    ->preload()
                    ->searchable()
                    ->required()
                    ->helperText('Only users who are not yet consultants should be selected.'),

                // NEW: Image Uploader
                FileUpload::make('image')
                    ->avatar() // Makes it round in preview
                    ->directory('consultants') // Saves in storage/app/public/consultants
                    ->image()
                    ->imageEditor(),

                Forms\Components\TextInput::make('specialization')
                    ->required(),

                Forms\Components\TextInput::make('experience_years')
                    ->numeric()
                    ->suffix('Years'),

                Forms\Components\Select::make('languages')
                    ->multiple()
                    ->options([
                        'English' => 'English',
                        'Hindi' => 'Hindi',
                        'Nepali' => 'Nepali',
                        'Sanskrit' => 'Sanskrit',
                    ]),

                Forms\Components\RichEditor::make('bio')
                    ->columnSpanFull(),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\TextColumn::make('user.name')->label('Consultant Name'),
                Tables\Columns\ImageColumn::make('image')->label('Profile Picture')->circular(),
                Tables\Columns\TextColumn::make('specialization'),
                Tables\Columns\TextColumn::make('experience_years')->suffix(' Years'),
                Tables\Columns\TextColumn::make('rating')
                    ->numeric(decimalPlaces: 1)
                    ->sortable()
                    ->label('Rating')
                    ->icon('heroicon-m-star')
                    ->color('warning'),
            ]);
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListConsultants::route('/'),
            'create' => Pages\CreateConsultant::route('/create'),
            'edit' => Pages\EditConsultant::route('/{record}/edit'),
        ];
    }
}
